/********************************************************************
 LPC812 - WS2812 demo using the SCT and SPI blocks

 Pins used in this application:
	
	P0.16 [O] - SCK0     : SPI0 SCK signal;  connect to SCT_IN0
	P0.13 [O] - MOSI0    : SPI0 MOSI signal; connect to SCT_IN1
	P0.16 [I] - SCT_IN0  : SCT sample clock input
	P0.13 [I] - SCT_IN1  : SCT data input
	P0.15 [O] - SCT_OUT0 : SCT data output - WS2812 DI0
********************************************************************/
#include "lpc8xx.h"

extern void SCT_Init(void);

volatile uint32_t led_data_cnt, led_data_index;
volatile uint8_t  led_data[30];                             // enough data for 10 LEDs: 10x3 = 30 bytes
volatile uint8_t  ev5_flag;

void SCT_IRQHandler (void)
{
    ev5_flag = 1;                                           // set flag
    LPC_SCT->EVFLAG = 0x3F;                                 // clear all SCT event flags
}

void SPI0_IRQHandler(void)
{
    LPC_SPI0->TXDAT = led_data[led_data_index];             // send next data
    led_data_index++;                                       // update the index
    if (led_data_cnt == led_data_index)
    {
        LPC_SPI0->INTENCLR = 1 << 1;                        // in case no more data left disable TXDAT interrupt
    }
}

void SPI0_Init(void)
{
    LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 11);                 // SPI0 clock enable

    LPC_SPI0->DIV    = (SystemCoreClock/800000)-1;          // SPI0 clock = 800 KHz
	LPC_SPI0->TXCTRL = (8-1) << 24 | 1 << 22;               // 8 bit frame, ignore RX data
    LPC_SPI0->CFG    = (1 << 0)  |                          // enable SPI0
                       (1 << 2)  |                          // master mode
                       (0 << 3)  |                          // MSB first
                       (1 << 4)  |                          // CPHA = 1
                       (1 << 5)  |                          // CPOL = 1
                       (0 << 7)  |                          // loopback mode disabled
                       (0 << 8);                            // SSEL polarity is active low

    NVIC_SetPriority(SPI0_IRQn, 0);                         // make sure this is the top priority isr
    NVIC_EnableIRQ(SPI0_IRQn);                              // Enable the SPI0 Interrupt
}

int main(void)
{
  uint32_t i, j;

    LPC_SYSCON->SYSAHBCLKCTRL |=  1 << 7;                   // enable SWM clock

    LPC_SWM->PINASSIGN3 &= ((16 << 24) | 0x00FFFFFF);       // SCK0     @ P0_16
    LPC_SWM->PINASSIGN4 &= ((13 << 0)  | 0xFFFFFF00);       // MOSI0    @ P0_13
    LPC_SWM->PINASSIGN5 &= ((16 << 24) | 0x00FFFFFF);       // SCT_IN0  @ P0_16
    LPC_SWM->PINASSIGN6 &= ((13 << 0)  | 0xFFFFFF00);       // SCT_IN1  @ P0_13
    LPC_SWM->PINASSIGN6 &= ((15 << 24) | 0x00FFFFFF);       // SCT_OUT0 @ P0_15

    SPI0_Init();                                            // setup SPI0
    SCT_Init();                                             // setup the SCT
	
    led_data_cnt = 9;
	
    while (1)
    {
        // turn off all LEDs
        led_data[0] = 0x00;                                 // LED0 G
        led_data[1] = 0x00;                                 // LED0 R
        led_data[2] = 0x00;                                 // LED0 B
        led_data[3] = 0x00;                                 // LED1 G
        led_data[4] = 0x00;                                 // LED1 R
        led_data[5] = 0x00;                                 // LED1 G
        led_data[6] = 0x00;                                 // LED2 G
        led_data[7] = 0x00;                                 // LED2 R
        led_data[8] = 0x00;                                 // LED2 B
        led_data_index = 0;                                 // reset data index
        while (!ev5_flag) ;                                 // make sure the done event occured
        ev5_flag = 0;                                       // clear flag
        LPC_SPI0->INTENSET = 1 << 1;                        // enable TXDAT based interrupt
        for (i = 0; i != 1000000; i++);                     // wait for a while

        // LED0 Red phase control (255 levels)
        for (j = 1; j != 256; j++)
        {
            led_data[0] = 0x00;                             // LED0 G
            led_data[1] = j;                                // LED0 R
            led_data[2] = 0x00;                             // LED0 B
            led_data[3] = 0x00;                             // LED1 G
            led_data[4] = 0x00;                             // LED1 R
            led_data[5] = 0x00;                             // LED1 G
            led_data[6] = 0x00;                             // LED2 G
            led_data[7] = 0x00;                             // LED2 R
            led_data[8] = 0x00;                             // LED2 B
            led_data_index = 0;                             // reset data index
            while (!ev5_flag) ;                             // make sure the done event occured
            ev5_flag = 0;                                   // clear flag
            LPC_SPI0->INTENSET = 1 << 1;                    // enable TXDAT based interrupt
            for (i = 0; i != 10000; i++);                   // wait for a while
        }
		
        // LED1 Green phase control (255 levels)
        for (j = 1; j != 256; j++)
        {
            led_data[0] = 0x00;                             // LED0 G
            led_data[1] = 0xFF;                             // LED0 R
            led_data[2] = 0x00;                             // LED0 B
            led_data[3] = j;                                // LED1 G
            led_data[4] = 0x00;                             // LED1 R
            led_data[5] = 0x00;                             // LED1 G
            led_data[6] = 0x00;                             // LED2 G
            led_data[7] = 0x00;                             // LED2 R
            led_data[8] = 0x00;                             // LED2 B
            led_data_index = 0;                             // reset data index
            while (!ev5_flag) ;                             // make sure the done event occured
            ev5_flag = 0;                                   // clear flag
            LPC_SPI0->INTENSET = 1 << 1;                    // enable TXDAT based interrupt
            for (i = 0; i != 10000; i++);                   // wait for a while
        }

        // LED2 Blue phase control (255 levels)
        for (j = 1; j != 256; j++)
        {
            led_data[0] = 0x00;                             // LED0 G
            led_data[1] = 0xFF;                             // LED0 R
            led_data[2] = 0x00;                             // LED0 B
            led_data[3] = 0xFF;                             // LED1 G
            led_data[4] = 0x00;                             // LED1 R
            led_data[5] = 0x00;                             // LED1 G
            led_data[6] = 0x00;                             // LED2 G
            led_data[7] = 0x00;                             // LED2 R
            led_data[8] = j;                                // LED2 B
            led_data_index = 0;                             // reset data index
            while (!ev5_flag) ;                             // make sure the done event occured
            ev5_flag = 0;                                   // clear flag
            LPC_SPI0->INTENSET = 1 << 1;                    // enable TXDAT based interrupt
            for (i = 0; i != 10000; i++);                   // wait for a while
        }

        // LED0/1/2 fade control (255 levels)
        for (j = 256; j != 0; j--)
        {
            led_data[0] = 0x00;                             // LED0 G
            led_data[1] = (j-1) << 0;                       // LED0 R
            led_data[2] = 0x00;                             // LED0 B
            led_data[3] = (j-1) << 0;                       // LED1 G
            led_data[4] = 0x00;                             // LED1 R
            led_data[5] = 0x00;                             // LED1 G
            led_data[6] = 0x00;                             // LED2 G
            led_data[7] = 0x00;                             // LED2 R
            led_data[8] = (j-1) << 8;                       // LED2 B
            led_data_index = 0;                             // reset data index
            while (!ev5_flag) ;                             // make sure the done event occured
            ev5_flag = 0;                                   // clear flag
            LPC_SPI0->INTENSET = 1 << 1;                    // enable TXDAT based interrupt
            for (i = 0; i != 10000; i++);                   // wait for a while
        }
    }
}
